/*
	Filename:	IFF.c
	Date:			20.96.11.25
	Compiler:	Borland C++ 5.0
	Target:		DOS Large

	EA IFF 85 File Support Routines
   (Electronic Arts Interchange File Format)

   NOTE: These routines handle Non-nested FORMs ONLY.

   Please refer to the "EA IFF 85" Standard for Interchange
   Format Files for a detailed dissertation on this standard.
*/

#include <stdio.h>
#include <alloc.h>

#define IFF_C
#ifndef IFF_H
#include "iff.h"
#endif

int IFF_Error = 0;

void *IFF_Malloc(int size);
void far *IFF_Farmalloc(ulong size);
struct IFF_Chunk *IFF_InsertChunk(struct IFF_ChunkList *CL,struct IFF_Chunk *Prev);
void IFF_DeleteChunk(struct IFF_ChunkList *CL,struct IFF_Chunk *C);
int IFF_Seek(FILE *fp,long nbytes,int offset);
int IFF_Read(FILE *fp,uchar *buf,long nbytes);
int IFF_Write(FILE *fp,uchar *buf,long nbytes);
int IFF_ReadChunkHeader(FILE *fp,struct IFF_ChunkHeader *CkHdr);
int IFF_WriteChunkHeader(FILE *fp,struct IFF_ChunkHeader *CkHdr);
int IFF_ReadChunk(FILE *fp,struct IFF_Chunk *Chk);
int IFF_WriteChunk(FILE *fp,struct IFF_Chunk *Chk);
int IFF_ReadFile(string filename,long ID,struct IFF_ChunkList *CL);
int IFF_WriteFile(string filename,long ID,struct IFF_ChunkList *CL);
int IFF_VerifyFormat(FILE *fp,long ID);

///////////////////////////////////////////////////////////////

void *IFF_Malloc(int size)
{
	/*
		Inputs:		Number of Bytes to allocate (< 64k)
		Output:		Pointer to allocated memory block
		Synopsis:	Wrapper for Standard Function malloc()
	*/
   char *mem;
   if(!(mem = malloc(size)))
   	IFF_Error = IFFERROR_MEMORY;
   return(mem);
}

///////////////////////////////////////////////////////////////

void far *IFF_Farmalloc(ulong size)
{
	/*
		Inputs:		Number of Bytes to allocate
		Output:		Pointer to allocated memory block
		Synopsis:	Wrapper for Standard Function farmalloc()
	*/
   char far *mem;
   if(!(mem = farmalloc(size)))
   	IFF_Error = IFFERROR_MEMORY;
   return(mem);
}

///////////////////////////////////////////////////////////////

struct IFF_Chunk *IFF_InsertChunk(struct IFF_ChunkList *CL,struct IFF_Chunk *Next)
{
	/*
		Inputs:		Pointer to IFF_ChunkList structure
      				Pointer to IFF_Chunk after one to be inserted
		Output:		Pointer to struct IFF_Chunk or NULL
		Synopsis:	Allocate and Insert Chunk into identified IFF_ChunkList.
      				If Next is NULL, then insert as Last in List.
	*/

   struct IFF_Chunk *C,*Prev;

   if(CL && ((C = (struct IFF_Chunk *)IFF_Malloc(sizeof(struct IFF_Chunk))) != NULL))
   {
   	/* Link into ChunkList */
      if(Next)
      {
      	Prev = Next->Prev;
         Next->Prev = C;
      }
      else
      {
      	Prev = CL->Last;
         CL->Last = C;
      }
      if(Prev)
      	Prev->Next = C;
      else
      	CL->First = C;
      C->Next = Next;
      C->Prev = Prev;
      setmem(C,sizeof(struct IFF_Chunk),0);
   }
   return(C);
}

///////////////////////////////////////////////////////////////

void IFF_DeleteChunk(struct IFF_ChunkList *CL,struct IFF_Chunk *C)
{
	/*
		Inputs:		Pointer to struct IFF_ChunkList structure
      				Pointer to struct IFF_Chunk structure
		Output:		None
		Synopsis:	Remove and Deallocate Chunk C in ChunkList CL
	*/

   if(CL && C)
   {
      struct IFF_Chunk *PC = C->Prev,*NC = C->Next;

      /* Detach from the List */
      if(PC)
      	PC->Next = NC;
      else
      	CL->First = NC;
      if(NC)
      	NC->Prev = PC;
      else
      	CL->Last = PC;
      if(C->Data) farfree(C->Data);
      free(C);
   }
}

///////////////////////////////////////////////////////////////

int IFF_Seek(FILE *fp,long nbytes,int offset)
{
	/*
		Inputs:		File Pointer
      				Number of Bytes to move
                  From where to start move:
					      SEEK_SET	0	File beginning
							SEEK_CUR	1	Current file pointer position
							SEEK_END	2	End-of-file
		Output:		1 for Success; 0 for Failure
		Synopsis:	Wrapper for Standard Function fseek()
	*/
   if(0 != fseek(fp,nbytes,offset))
   {
   	IFF_Error = IFFERROR_SEEK;
      return(0);
   }
   return(1);
}

///////////////////////////////////////////////////////////////

int IFF_Read(FILE *fp,uchar *buf,long nbytes)
{
	/*
		Inputs:		File Pointer
      				Pointer to buffer
                  Number of Bytes to read into buffer
		Output:		1 for Success; 0 for Failure
		Synopsis:	Wrapper for Standard Function fread()
	*/

   if(1 > fread(buf,nbytes,1,fp))
   {
   	IFF_Error = IFFERROR_READ;
      return(0);
   }
   return(1);
}

///////////////////////////////////////////////////////////////

int IFF_Write(FILE *fp,uchar *buf,long nbytes)
{
	/*
		Inputs:		File Pointer
      				Pointer to buffer
                  Number of Bytes to read into buffer
		Output:		1 for Success; 0 for Failure
		Synopsis:	Wrapper for Standard Function fwrite()
	*/

   if(1 > fwrite(buf,nbytes,1,fp))
   {
   	IFF_Error = IFFERROR_WRITE;
      return(0);
   }
   return(1);
}

///////////////////////////////////////////////////////////////

int IFF_ReadChunkHeader(FILE *fp,struct IFF_ChunkHeader *CkHdr)
{
	/*
		Inputs:		File Pointer
      				Pointer to ChunkHeader Structure
		Output:   	1 for Success; 0 for Failure
		Synopsis:	Read Chunkheader information
	*/
    return(IFF_Read(fp,(uchar *)CkHdr,sizeof(struct IFF_ChunkHeader)));
}

///////////////////////////////////////////////////////////////

int WriteChunkHeader(FILE *fp,struct IFF_ChunkHeader *CkHdr)
{
	/*
		Inputs:		File Pointer
      				Pointer to ChunkHeader Structure
		Output:   	1 for Success; 0 for Failure
		Synopsis:	Write Chunkheader information
	*/
   return(IFF_Write(fp,(uchar *)CkHdr,sizeof(struct IFF_ChunkHeader)));
}

///////////////////////////////////////////////////////////////

int IFF_ReadChunk(FILE *fp,struct IFF_Chunk *Chk)
{
	/*
		Inputs:		File Pointer
      				Pointer to IFF_Chunk structure
		Output:		1 for Success; 0 for Failure
		Synopsis:	Read Chunk
	*/
   if(IFF_ReadChunkHeader(fp,(struct IFF_ChunkHeader *)Chk))
   {
   	if(IFF_Read(fp,Chk->Data,Chk->Size))
      {
	      /* Skip Pad Byte */
		   if(Chk->Size & 1)
		      return(IFF_Seek(fp,1,SEEK_CUR));
         else
         	return(1);
      }
   }
   return(0);
}

///////////////////////////////////////////////////////////////

int IFF_WriteChunk(FILE *fp,struct IFF_Chunk *Chk)
{
	/*
		Inputs:		File Pointer
      				Pointer to IFF Chunk Structure
		Output:		1 for Success; 0 for Failure
		Synopsis:	Write Chunk data
	*/
   if(IFF_WriteChunkHeader(fp,(struct IFF_ChunkHeader *)Chk))
   {
	   if(IFF_Write(fp,Chk->Data,Chk->Size))
      {
		   if(Chk->Size & 1)
		   {
		   	/* Write Pad Byte */
		      uchar Zero = 0;
		      return(IFF_Write(fp,&Zero,1L));
         }
         else
         	return(1);
      }
   }
   return(0);
}

///////////////////////////////////////////////////////////////

int IFF_ReadFile(string filename,long ID,struct IFF_ChunkList *CL)
{
	/*
		Inputs:		Filename String
      				FORM Type ID
   					Pointer to ChunkList to hold Chunks
		Output:		1 for Success; 0 for Failure
		Synopsis:	Read IFF File Information and Data

      NOTE: In future, add feature to terminate read on user-specified
      		Chunk ID as well as read entire file.
	*/
	FILE *fp = NULL;
   int Status = 0;

   if(CL)
   {
	   if((fp = fopen(filename,"rb")) != NULL)
	   {
	      /* Determine that file is IFF FORM of type ID before proceeding */
	      if(IFF_VerifyFormat(fp,ID))
	      {
	      	while(!feof(fp))
	         {
	         	/* Append Chunk to end of ChunkList */
	         	if(!IFF_InsertChunk(CL,NULL))
	            	break;
	           	if(!IFF_ReadChunk(fp,CL->Last))
	              	break;
	         }
	         /* Successfully reached End Of File */
	         if(feof(fp))
	         	Status = 1;
	      }
	      else if(!IFF_Error)
	      	IFF_Error = IFFERROR_FORM;		/* Not of type ID */
	      fclose(fp);
	   }
		else
	   	IFF_Error = IFFERROR_OPEN;
   }
   return(Status);
}

///////////////////////////////////////////////////////////////

int IFF_WriteFile(string filename,long ID,struct IFF_ChunkList *CL)
{
	/*
		Inputs:		Filename String
      				FORM Type ID
      				Pointer to ChunkList holding information
		Output:		1 for Success; 0 for Failure
		Synopsis:	Write IFF File Information and Data
	*/

   FILE *fp;
   struct IFF_Chunk Chk;
   struct IFF_Chunk *C;
   int Status = 0;

   if(CL)
   {
	   if((fp = fopen(filename,"wb")) != NULL)
	   {
	      /* Write FORM Header and Form Type */
	      /* Size will be appended at end of Chunk writes */
	      Chk.ID = ID_FORM;
	      Chk.Size = 4L;
         Chk.Data = &ID;
	      if(IFF_WriteChunk(fp,&Chk))
	      {
            for(C = CL->First; C != NULL; C = C->Next)
            {
	         	Chk.Size += AlignedSize(C->Size);
               if(!IFF_WriteChunk(fp,C))
               	break;
            }
            if(C == NULL)
            {
			      /* Append FORM size */
			      if(IFF_Seek(fp,IDSIZE,SEEK_SET))
               {
				      if(IFF_Write(fp,(uchar *)&Chk.Size,4L))
			      		Status = 1;
               }
	         }
	      }
	      fclose(fp);
	   }
	   else
	   	IFF_Error = IFFERROR_OPEN;
   }
	return(Status);
}

///////////////////////////////////////////////////////////////

int IFF_VerifyFormat(FILE *fp,long ID)
{
	/*
		Inputs:		File Pointer
                  FORM Type Identification
		Output:		1 for Success; 0 for Failure
		Synopsis:	Verify IFF Format and Type
	*/

	struct IFF_ChunkHeader CkHdr;

	if(IFF_ReadChunkHeader(fp,&CkHdr))
   {
		if(CkHdr.ID == ID_FORM)
	   {
	   	if(IFF_Read(fp,(uchar *)&CkHdr.ID,IDSIZE))
         {
         	if(CkHdr.ID == ID)
            	return(1);
	      }
      }
	   else if((CkHdr.ID == ID_LIST) || (CkHdr.ID == ID_CAT))
      	IFF_Error = IFFERROR_NESTED;
      else
      	IFF_Error = IFFERROR_FORMAT;		/* Not an IFF File */
   }
   return(0);
}

